/* ****************************************************** */
/* nesting.c                                              */
/* Count brace nesting in TeX files                       */
/*                                                        */
/* Author:    vodoo@vakw.ch                               */
/* Copyright: 2012 vodoo@vakw.ch                          */
/* License:   GPLv2                                       */
/* ****************************************************** */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <locale.h>

#define  BUFSIZE 4096
#define  DEBUG   0

unsigned char linebuf[BUFSIZE+1];

int lbp      = 0;  /* pointer into buffer         */
int escape   = 0;  /* escape mode on/off switch   */
int comment  = 0;  /* comment mode switch         */
int premath  = 0;  /* single dollar sign flag     */
int mathmode = 0;  /* math mode on/off switch     */
int dispmode = 0;  /* math display mode switch    */
int verbmode = 0;  /* verbatim mode switch        */
int ignoreve = 0;  /* ignore verbatim char switch */
int para     = 0;  /* detects paragraph end \n\n  */

unsigned char vc = '|';   /* default verbatim mode char   */
                          /* as used in the TeXbook       */

int main(int argc, char *argv[]) {

  FILE *infile;
  int c;           /* input character from file */
  int nl = 0;      /* nesting level counter     */
  int nlup = 0;
  int nldown = 0;
  int cleanup = 0; /* cleanup switch            */
  char *counted = "counted";
  char *ignored = "ignored";
  char *message;

  setlocale(LC_ALL, "");

  /* Check for required file argument */
  if(argc < 2) {
    fprintf(stderr, "%s: %s\n%s %s %s\n", argv[0], "Missing file argument",
      "Usage:", argv[0], "<filename>");
    exit(1);
  }

  if((argc >= 2) && ((strncmp(argv[1], "-h", 2)        == 0)
                 ||  (strncmp(argv[1], "--help", 6)    == 0)
                 ||  (strncmp(argv[1], "-v", 2)        == 0)
                 ||  (strncmp(argv[1], "--version", 9) == 0))) {
    printf("%s\n", argv[0]);
    printf("Version: 1.1\n");
    printf("Usage:   nesting <filename> [--verbatim <char>]\n");
    printf("         nesting -h | --help | -v | --version\n\n");
    printf("Show the contents of the input file on STDOUT, each line preceeded\n");
    printf("by a number indicating the nesting level of braces '{}' valid at\n");
    printf("the end of the respective line.\n\n");
    printf("When the filename is - input is read from stdin.\n\n");
    printf("TeX syntax is used. Braces in comments are not taken into account.\n");
    printf("A special character can be defined to start/end verbatim mode.\n");
    printf("Default is '|'. Use --verbatim '' to disable it.\n");
    return(0);
  }

  /* Replace the default verbatim character */
  if((argc >= 4) && (strncmp(argv[3], "--verbatim", 10) == 0)) {
    vc = argv[4][0];
  }
    
  /* Open the input file for read */
  if(!strncmp(argv[1], "-", 1)) {
    infile = stdin;
  } else {
    if((infile = fopen(argv[1], "r")) == NULL) {
      fprintf(stderr, "%s %s\n", "Error: could not open file", argv[1]);
      exit(1);
    }
  }

  /* Read from file */
  while((c = fgetc(infile)) != EOF) {

    /* Complete dangling mathmode settings for a single $ */
    if((premath) && (c != '$') && (!comment)) {
      premath = 0;
      if(mathmode) {
        mathmode = 0;
#if DEBUG
        printf("***  Math mode terminated on single $\n");
#endif
      } else {
        mathmode = 1;
#if DEBUG
        printf("***  Math mode started on single $\n");
#endif
      }
    }

    /* Count braces but ignore escaped braces in mathmode */
    if((!comment) && (!verbmode)) {
      if((mathmode) || (dispmode)) {
        if(!escape) {
          if(c == '{') {
            ++nl;
            ++nlup;
          }
          if(c == '}') {
            --nl;
            ++nldown;
          }
        }
      } else { /* not in mathmode */
        if(c == '{') {
          ++nl;
          ++nlup;
        }
        if(c == '}') {
          --nl;
          ++nldown;
        }
      }
    }

#if DEBUG
    if(comment || verbmode || ((mathmode || dispmode) && escape)) {
      message = ignored;
    } else {
      message = counted;
    }

    if(c == '{') {
      printf("***  { found co=%1d ve=%1d ma=%1d md=%1d esc=%1d %s\n", comment,
        verbmode, mathmode, dispmode, escape, message);
    }

    if(c == '}') {
      printf("***  } found co=%1d ve=%1d ma=%1d md=%1d esc=%1d %s\n", comment,
        verbmode, mathmode, dispmode, escape, message);
    }
#endif

    /* Show counter and input line when reaching the end of the line */
    if(c == '\n') {
      if(para) {
        verbmode = 0;
      }
      para = 1;
      cleanup = 0;
      linebuf[lbp] = '\0';
      lbp = 0;
      comment = 0;

      /* For the TeXbook */
      if(!strncmp(linebuf, "\\begintt", 8)) {
        verbmode = 1;
        ignoreve = 1;
      }

      /* For the TeXbook */
      if(!strncmp(linebuf, "\\endtt", 6)) {
        verbmode = 0;
        ignoreve = 0;
      }

      /* Ugly hack for a case which occurs once in The TeXbook */
      /* \endtt should be on a line by itself to work properly */
      if(!strncmp(linebuf, "\\endtt}", 7)) {
        --nl;
      }

      /* Ugly hack #2: do not miscount the \verbatim definition */
      if((strstr(linebuf, "endgroup\\fi\\fi\\fi\\next>>")) && (!verbmode)) {
        nl = nl - 2;
      }

#if DEBUG
      if(verbmode) {
      printf("%3dv %s\n", nl, linebuf);
      } else {
      printf("%3d  %s\n", nl, linebuf);
      }
#else
      printf("%3d %s\n", nl, linebuf);
#endif
      nlup = 0; nldown = 0;

    } else {
      /* store in buffer */
      para = 0;
      if(lbp < BUFSIZE) {
        if(!(c == '\014')) {
          linebuf[lbp++] = (unsigned char) c;
          cleanup = 1;
        }
      } else {
        linebuf[lbp] = '\0';
        printf("%3d %s\n", nl, linebuf);
        fflush(stdout);
        fclose(infile);
        fprintf(stderr, "%s\n", "Error: line too long; buffer space is full");
        exit(1);
      }
    }

    /* % starts a comment unless preceded by an escape character */
    if((c == '%') && (!comment) && (!escape) && (!verbmode)) {
      comment = 1;
    }

    /* Detect start and stop of verbatim mode */
    if((c == vc) && (!comment) && (!ignoreve)) {
      if(verbmode) {
        verbmode = 0;
#if DEBUG
        printf("***  Terminating verbatim mode\n");
#endif
      } else {
        if(!escape)
          verbmode = 1;
#if DEBUG
          printf("***  Entering verbatim mode\n");
#endif
      }
    }

    /* Math mode starts and ends with $ or $$ */
    /* ex.  |\cases| ( $\bigl\{{\cdots\atop\cdots}$ ), 175, +362. */
    if((c == '$') && (!comment) && (!verbmode) && (!escape)) {
      if(dispmode) {
        if(premath) {
          dispmode = 0;
          if(mathmode) {
            mathmode = 0;
#if DEBUG
            printf("***  WARNING terminating dangling math mode\n");
#endif
          }
          premath  = 0;
#if DEBUG
          printf("***  Math display mode terminated on double $$\n");
#endif
        } else {
          premath =  1;
        }
      } else {
        if(premath) {
          dispmode = 1;
          premath  = 0;
#if DEBUG
          printf("***  Math display mode started on double $$\n");
#endif
        } else {
          premath =  1;
        }
      }
    }

    /* \ starts escaping but is ignored in comments and verbatim mode */
    if((c == '\\') && (!verbmode)) {
      if(!comment) {
        if(escape) {
          escape = 0;
        } else {
          escape = 1;
        }
      }
    } else { /* any other character resets escape mode */
      escape = 0;
    }

  } /* End of while loop */

  fclose(infile);

  /* print unfinished line (there is no \nl at the end of the file) */
  if(cleanup) {
    linebuf[lbp] = '\0';
    printf("%3d %s\n", nl, linebuf);
  }

  fflush(stdout);

  if(mathmode) {
    fprintf(stderr, "%s\n", "Warning: file ended in math mode");
  }

  if(nl > 0 || nl < 0) {
    fprintf(stderr, "%s %d\n", 
      "Warning: file ended with brace nesting of level", nl);
  }

  fflush(stderr);
  return(0);
}
