/* collatz.c

    Version:	1.0.6
    Date:	2023-01-30
    Licence:	Open source / public domain

    Calculate a sequence of the Collaz conjecture

    For each n: do
    If n is even:	n = n/2
    and if n is odd:	n = 3n+1

    Does this sequence always end at the 1-4-2-1 loop ???
    Computations up to 2^68 have shown: yes, at least up to this value of n.
    (https://web.archive.org/web/20211117141656/https://pcbarina.fit.vutbr.cz/path-records.htm)

    Actually newer work shows convergence to 1 up to 492807256 * 2^40
    = 541847308224383942656.

    Usage:
        collatz [gmp] [srec <int>] [mrec <int>] <start> <stop> [list]
            Compute the sequence from n=start to n=stop
            Print new records for the numbers of steps used to reach 1
            and for the highest integer reached in the process.

        collatz [gmp] <start> show
            Compute the sequence for n=start and print
            the value of n at each step.

    Technical stuff:
    You can start this program on a specific cpu core using 'taskset'.
    Or you could amend this code using the 'sched_setaffinity' function
    (see 'man 3 sched_setaffinity' and/or 'man 3 CPU_SET').
    Use 'cat /proc/cpuinfo' to see how many CPU cores you have.
*/

#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <stdbool.h>
#include <string.h>
#include <gmp.h>
#include <assert.h>
#include <time.h>
#include <sys/time.h>

#define KEEP_BATCH_MAXIMUM
// #define USE_VERBOSE_MODE

/**
* This nice subroutine was stolen from:
* https://stackoverflow.com/questions/10098441/get-the-current-time-in-milliseconds-in-c
*/

int64_t currentTimeMillis() {
  struct timeval time_of_day;
  gettimeofday(&time_of_day, NULL);
  int64_t s1 = (int64_t)(time_of_day.tv_sec) * 1000;
  int64_t s2 = (time_of_day.tv_usec / 1000);
  return s1 + s2;
}


int main(int argc, char *argv[]) {

    unsigned long counter;
    unsigned long n;
    unsigned long n_temp;
    unsigned long start = 2UL;
    unsigned long stop = 10000000UL;
    unsigned long record_top = 0UL;
    unsigned long record_steps = 0UL;
    unsigned long srec_record_steps = 0UL;
    unsigned long steps_bucket = 0UL;
    unsigned long max_value = 1UL;
#ifdef KEEP_BATCH_MAXIMUM
    unsigned long batch_max_value_n = 1UL;
#endif
    unsigned long max_n = ULONG_MAX; /* = 18446744073709551615 */
    unsigned long limit_n; /* limit before n*3 + 1, 6148914691236517204UL */

    /* GCC math bails out to GMP at N > limit_n to avoid overflow */
    limit_n = (max_n - 1UL) / 3UL;

    /* Count steps required for n to reach 1, <= 4294967295U
       This is also called 'total stopping time'.
       There is no reason for a long int here. Reaching this boundary is a clear indication
       that something went wrong or that a circular sequence other than 1 4 2 was found.

       Funny note: N=7219136416377236271195 takes 2968 steps to get to 1 and the maximum
       value of n is reached at step 231; Running the collatz sequence on the first 10000
       digits of Pi (omitting the decimal point after the leading 3) goes to 1 after
       239030 steps.

       N=3716509988199 goes up to 207936463344549949044875464 (802 steps to 1).
       N=3743559068799 needs 1549 steps to 1
       N=9016346070511 goes up to 252229527183443335194424192 (27 digits, 886 steps to 1).
    */

    unsigned int steps;
    unsigned int max_steps = 4294967295U; /* This is UINT_MAX */

    /* Experimental: increment between subsequent values of n to test */
    unsigned long inc;
    unsigned long remainder;
    inc = 1UL;

    /* Counter to show progress and speed after limit_i calculations of n */
    /* limit_i will be divided by 5 when using GNU MP mode */
    int64_t timer_start;
    int64_t timer_stop;
    int64_t time_elapsed;
    unsigned int i = 1U;
    unsigned int limit_i = 10000000U;
    unsigned int gmp_used = 0U;
    unsigned int limit_i_divisor = 5U;
    
    /* Input arg conversion assumes BASE 10 */
    int base = 10;
    char *endptr;
    const char * gmptxt = "GMP";
    int offset = 0;
    /* For the length of a input string */
    size_t sl = 0;

    bool show_it         = false;
    bool step_new_record = false;
    bool max_new_record  = false;
    bool show_steps      = false;
    bool printhelp       = false;
    bool use_gmp         = false;
    bool list_all        = false;		/* True means: List all results, not just records */
    bool run             = true;
    bool use_gcc_loop    = true;
    bool show_anyway     = false;
    bool show_gmp_flag   = true;

    /* Check for completely missing <args>
        or for a cry for help.
       Do this before initializing mpz_t variables
    */

    if (argc == 1) {
        printhelp = true;
    }

    if ((argc == 2) && 
        ((strcmp(argv[1], "-h") == 0) ||
        (strcmp(argv[1], "-help") == 0) ||
        (strcmp(argv[1], "--help") == 0) ||
        (strcmp(argv[1], "help") == 0))) {
        printhelp = true;
    }

    if (printhelp) {
        printf("\nCalculate a sequence of the Collaz conjecture\n\n");
        printf("\tFor each n = any integer > 2 do the following\n");
        printf("\t\tIf n is even:       take n = n/2\n");
        printf("\t\tand if n is odd:    take n = 3n+1\n");
        printf("\tThen repeat\n\n");
        printf("\tDoes this sequence always end at n equals 1 ?\n\n");
        printf("%s usage:\n", argv[0]);
        printf("\t%s [help|-help|--help]\tShow this help page, or\n\n", argv[0]);
        printf("\t%s [gmp] [srec <int>] [mrec <int>] <start> <stop>\n", argv[0]);
        printf("\t\tgmp\t(optional). Force use of the GNU MP library\n");
        printf("\t\t\tfor the entire computation. This is ~16 times slower\n");
        printf("\t\t\tbut allows for arbitrarily large numbers without\n");
        printf("\t\t\tthe constraints mentioned below for <start> and <stop>.\n\n");
        printf("\t\tsrec <integer>\tSet previous record for number of steps.\n");
        printf("\t\t\tUseful when <start> is bigger than 2. The number of steps is\n");
        printf("\t\t\tnot a new record while not bigger than <int>.\n\n");
        printf("\t\tmrec <integer>\tSet previous record for the maximum reached\n");
        printf("\t\t\tduring a sequence. New record is not shown while the maximum\n");
        printf("\t\t\tof any sequence is not bigger than <int>.\n\n");
        printf("\t\tstart\tStarting point n for the computation of (n*3 + 1).\n");
        printf("\t\t\tAny positive integer n >= 2 and <= 18446744073709551615.\n");
        printf("\t\t\t<start> is incremented up to <stop> for each n-sequence.\n\n");
        printf("\t\tstop\tStop computation after n = stop has been reached.\n");
        printf("\t\t\tAny positive integer >= <start> and <= 18446744073709551615.\n\n");
        printf("\t%s [gmp] <start> show\n", argv[0]);
        printf("\t\t\tCompute the sequence for n=start, show all intermediate\n");
        printf("\t\t\tsteps of n on it's way to 1, then stop.\n\n");
        printf("\t%s [gmp] <start> <stop> [list]\n", argv[0]);
        printf("\t\tlist\t(optional). List results for every n, not just records.\n\n");
        fflush(stdout);
        exit (EXIT_SUCCESS);
    }


    /* Variables for GMP */

    mpz_t big_n;
    mpz_t big_n_temp;
    mpz_t big_start;
    mpz_t big_stop;
    mpz_t big_max_value;
#ifdef KEEP_BATCH_MAXIMUM
    mpz_t batch_max_value;
    mpz_t big_batch_max_value_n;
#endif
    mpz_t big_record_top;
    mpz_t big_difference;
    mpz_t big_remainder;
    mpz_t big_one;
    mpz_t big_two;
    mpz_t big_three;

    mpz_init(big_n);
    mpz_init(big_n_temp);
    mpz_init(big_start);
    mpz_init(big_stop);
    mpz_init(big_max_value);
#ifdef KEEP_BATCH_MAXIMUM
    mpz_init(batch_max_value);
    mpz_init(big_batch_max_value_n);
#endif
    mpz_init(big_record_top);
    mpz_init(big_difference);
    mpz_init(big_remainder);
    mpz_init(big_one);
    mpz_init(big_two);
    mpz_init(big_three);

    mpz_set_ui(big_one, 1UL);
    mpz_set_ui(big_two, 2UL);
    mpz_set_ui(big_three, 3UL);


    /* Initialize batch_max_value to 1 */
#ifdef KEEP_BATCH_MAXIMUM
    mpz_set_ui(batch_max_value, 1UL);
    mpz_set_ui(big_batch_max_value_n, 1UL);
#endif


    /* ========== RECORDS ========== */

    /* Users can set some hard wired values here (not recommended) */
    /* Records set on the command line with
       srec <INT> or mrec <INT> take precedence. */
    
    /* Set record steps and record top here for gcc math when you don't start at 2 */
    /* Statistically it takes 9.477955 steps to get below the point where you started */
    record_steps = 1UL;

    /* Set record_top for gcc and GMP */
    /* This is also called the 'height' on the calculation */
    /* Example: (Please no more than 18446744073709551615 for C math) */
    /* record_top       =  18144594937356598024UL; */
    record_top   = 1UL;	/* my choice, set your own as desired */

    /* Same for GMP: */
    /* Enter your number as a character string for GMP, this is just an example: */

    /*
    const char * brt = "68838156641548227040";
    mpz_set_str(big_record_top, brt, base);
    */

    mpz_set_ui(big_record_top, 1UL);		/* my choice, set your own as desired or comment out */

    /* ======= END OF RECORDS ====== */



    /* Process command line options
        This is a mess; you are not expected to understand this */

    /* Look for the gmp flag */
    if (argc >= 2) {
        if (strcmp(argv[1], "gmp") == 0) {
            offset = 1;
            use_gmp = true;
            /* Show time benchmarks more often */
            limit_i = limit_i / limit_i_divisor;  /* GNU MP mode is 10 times slower */
            printf ("Using GNU MP library\n");
        } else {
            printf ("GNU MP library may be used automatically when required\n");
        }
    }

    /* Get initial value for record_steps */
    if (argc >= (2+offset)) {
        if (strcmp(argv[1+offset], "srec") == 0) {
            offset++;
            errno = 0;
            srec_record_steps = strtoul(argv[1+offset], &endptr, base);
            if ((errno > 0) || (srec_record_steps > max_steps)) {
                printf("Oops ... something with <int> argument for 'srec' went wrong. Error is %d, <int> is %lu\n", errno, srec_record_steps);
                fflush(stdout);
                return errno;
            } else {
                printf("Got input record steps   = %lu\n", srec_record_steps);
                record_steps = srec_record_steps;
                offset++;
            }
        }
    }

    /* Get initial value for record_top */
    if (argc >= (2+offset)) {
        if (strcmp(argv[1+offset], "mrec") == 0) {
            offset++;
            if (mpz_set_str(big_record_top, argv[1+offset], base) == -1) {
                printf("Oops ... something with <int> argument for 'mrec' went wrong. Error is %d, <int> is %s\n", errno, argv[1+offset]);
                fflush(stdout);
                return errno;
            }

            if (mpz_cmp_ui(big_record_top, max_n) <= 0) {
                errno = 0;
                record_top = strtoull(argv[1+offset], &endptr, base);
                if (errno > 0) {
                    printf("Oops ... something with <int> argument for 'mrec' went wrong. Error is %d, <int> is %lu\n", errno, record_top);
                    fflush(stdout);
                    return errno;
                } else {
                    printf("Got input record top gcc = %lu\n", record_top);
                }
            } else {
                record_top = max_n;
            }
            printf("Got input record top gmp = ");
            mpz_out_str(stdout, 10, big_record_top);
            printf("\n");
            offset++;
        }
    }

    /* Fetch <start> */
    if (argc >= (2+offset)) {
        if (use_gmp == false) {
            errno = 0;
            sl = strlen(argv[1+offset]);
            start = strtoull(argv[1+offset], &endptr, base);
            if ((errno > 0) || (start < 2)) {
                printf("Oops ... something with argument 'gmp' or <start> went wrong. Error is %d, <start> is %lu\n", errno, start);
                if (sl > 19) {
                    printf("Your input is %lu bytes long. Please consider to use the 'gmp' option\n", sl);
                }
                fflush(stdout);
                return errno;
            }
            printf("Got this number to start with: %lu\n", start);
        } else {
            /* allow huge numbers for <start> */
            mpz_set_str(big_start, argv[1+offset], base);
            printf("Got this number to start with: ");
            mpz_out_str(stdout, 10, big_start);
            printf("\n");
        }
    }

    /* Fetch 'show' or <stop> */
    if (argc >= (3+offset)) {
        if (strcmp(argv[2+offset], "show") == 0) {
            if (use_gmp == false) {
                stop = start;
            } else {
                mpz_set(big_stop, big_start);
            }
            inc = 1UL; /* Bigger inc wouldn't make sense */
            show_steps = true;
        } else {
            if (use_gmp == false) {
                errno = 0;
                sl = strlen(argv[2+offset]);
                stop = strtoull(argv[2+offset], &endptr, base);
                if (errno > 0) {
                    printf("Oops ... something with argument <stop> or 'show' went wrong. Error is %d\n", errno);
                    if (sl > 19) {
                        printf("Your input is %lu bytes long. Please consider to use the 'gmp' option\n", sl);
                    }
                    fflush(stdout);
                    return errno;
                } else {
                    printf("Got this number to stop at:    %lu\n", stop);
                }
            } else {
                /* allow huge numbers for <stop> */
                mpz_set_str(big_stop, argv[2+offset], base);
                printf("Got this number to stop at:    ");
                mpz_out_str(stdout, 10, big_stop);
                printf("\n");
            }
        }
    }

    /* Fetch optional argument 'list' if present */
    if (argc >= (4+offset)) {
        if (strcmp(argv[3+offset], "list") == 0) {
            list_all = true;
            printf("Listing all results\n");
        }
    }

    printf("\n");
    fflush(stdout);

    /* 	Calculation of all sequences for n = <start> up to and including n = <stop>.
        gcc unsigned long int will be used.

        However, the calculation will automatically switch to using gmp big numbers
        to avoid overflow when necessary.

        The initial value of n must remain below ULONG_MAX.
    */

    timer_start = currentTimeMillis();


    /* ================================================================
       Calculation without permanent use of GNU MP library gmp selected
       ================================================================ */

    if (! use_gmp) {

        /* Adjust <stop> to be hit when using increments > 1 */
        if ((inc > 1UL) && ((stop - start) >= inc)) {
            remainder = (stop - start) % inc;
            if (remainder > 0UL) {
                /* incrementing <start> will not hit the <stop> value> */
                if (max_n >= (stop + (inc - remainder))) {
                    /* We can increment <stop> without overflow */
                    stop += (inc - remainder);
                } else {
                    stop -= remainder;
                }
            }
        } else {
            if (inc > 1UL) {
                printf("Error: selected increment %lu is too big to fit between <start> and <stop>\n", inc);
                fflush(stdout);
                exit(EXIT_FAILURE);
            }
        }

        /* Ignore skipping of some N values to round reporting boundaries */
        if ((start < 100UL) && (stop >= limit_i)) {
            i = i + start;
        }

        /* run is required to stop the loop when limit is reached in gmp calculation
           and start can't be incremented above stop without overflow */
        while ((start <= stop) && run) {
            n = start;
            max_value    = 1UL;
            steps        = 0U;
            show_it      = false;

            if (show_steps) {
                printf("    Steps for calculation of n=%lu:\n", n);
            }

            while (n > 1UL && use_gcc_loop) {
                /* Using normal gcc math */
                if (n & 1) {
                    /* odd number */
                    if (n > limit_n) {
                        /* This is e.g. triggered by start=12327829503 at step 201 */
#ifdef USE_VERBOSE_MODE
                        printf("limit_n is reached, calculation is using GMP for n=%lu\n", start);
                        fflush (stdout);
                        show_anyway  = true;
#endif
                        use_gcc_loop = false;
                        record_top   = max_n; /* = ULONG_MAX, avoid record_top to be used ever again */
                        /* Exit this gcc math loop temporarily, use gmp math instead */
                        break;
                    } else {
                        n = (n * 3UL) + 1UL;
                        /* Avoid the multiplication. However, it's not faster :-) */
                        /* n += ((n<<1) | 1UL); */
                    }
                } else {
                    /* even number; divide by 2 */
                    n = n>>1;
                }

                steps++;

                /* There is no real need for this test.
                   Numerical limit for steps will not be exceeded
                   within gcc math range.
                   However, uncommenting it will not slow
                   down the calculation noticeably. */

#ifdef USE_VERBOSE_MODE
                if (steps == max_steps) {
                    printf("Stopped at steps=%u because steps reached numerical limit\n", steps);
                    printf("Please check what is going on: collatz %lu show\n", start);
                    fflush(stdout);
                    mpz_clear(big_n);
                    mpz_clear(big_start);
                    mpz_clear(big_stop);
                    mpz_clear(big_record_top);
                    mpz_clear(big_max_value);
                    mpz_clear(big_one);
                    mpz_clear(big_two);
                    mpz_clear(big_three);
#ifdef KEEP_BATCH_MAXIMUM
                    mpz_clear(batch_max_value);
                    mpz_clear(big_batch_max_value_n);
#endif
                    exit(EXIT_FAILURE);
                }
#endif

                if (show_steps) {
                    printf("    Step %u:\tn=%lu\n", steps, n);
                }

                /* Remember the record for n */
                if (n > max_value) {
                    max_value = n;
                }
            } /* End of inner while loop using gcc math */


            /* Same as above, but use GMP for calculation for n=start.
               Here we do the interrupted inner loop again for n with GMP.
               Keep track of the maximum within this batch from start to stop. */

            if (! use_gcc_loop) {
                mpz_set_ui(big_n, start);
                mpz_set_ui(big_max_value, 1UL);
                steps = 0U;
                show_it = false;
                gmp_used += 1;

                while (mpz_cmp(big_n, big_one) > 0) { /* while big_n > 1 */
                    /* if (mpz_divisible_ui_p(big_n, 2UL)) { */
                    if (mpz_divisible_2exp_p(big_n, 1)) {
                        /* Division by 2 */
                        /* mpz_divexact_ui(big_n, big_n, 2UL); */

                        /* Division by 2 doing 1 bitshift right; is a bit faster */
                        mpz_tdiv_q_2exp(big_n, big_n, 1);
                    } else {
                        /* big_n = big_n * 3 + 1 */
                        mpz_mul(big_n, big_n, big_three);
                        mpz_add(big_n, big_n, big_one);
                    }

                    steps++;

                    if (steps == max_steps) { /* = 4294967295 */
                        printf("Stopped at steps=%u because steps reached numerical limit\n", steps);
                        printf("Please check what is going on: collatz %lu show\n", start);
                        mpz_clear(big_n);
                        mpz_clear(big_start);
                        mpz_clear(big_stop);
                        mpz_clear(big_record_top);
                        mpz_clear(big_max_value);
#ifdef KEEP_BATCH_MAXIMUM
                        mpz_clear(batch_max_value);
                        mpz_clear(big_batch_max_value_n);
#endif
                        mpz_clear(big_difference);
                        mpz_clear(big_remainder);
                        mpz_clear(big_one);
                        mpz_clear(big_two);
                        mpz_clear(big_three);
                        fflush(stdout);
                        exit(EXIT_FAILURE);
                    }

                    if (show_steps) {
                        printf("    Step %u:\tn=", steps);
                        mpz_out_str(stdout, 10, big_n);
                        printf ("\n");
                    }

                    /* Remember the record for n */
                    if (mpz_cmp(big_n, big_max_value) > 0) {
                        mpz_set(big_max_value, big_n);
                    }

                } /* End of inner while loop using gmp */
            } /* End of alternative using gmp */


            /* Continuation: common code for gcc and GMP math */

            /* steps is UINT in both modes of calculation */
            steps_bucket += steps;

            if (steps > record_steps) {
                show_it         = true;
                record_steps    = steps;
                step_new_record = true;
            }

            /* Check for new MAX record */
            if (use_gcc_loop) {
                if (max_value > record_top) {
                    show_it        = true;
                    record_top     = max_value;
                    max_new_record = true;
                }
            } else { /* GMP was used */
                if (mpz_cmp(big_max_value, big_record_top) > 0) {
                    show_it    = true;
                    mpz_set(big_record_top, big_max_value);
                    max_new_record = true;

                    /* Backport the record to max_value if possible */
                    if (record_top < max_n) {
                        if (mpz_cmp_ui(big_max_value, max_n) <= 0) {
                            max_value  = mpz_get_ui(big_max_value);
                            record_top = max_value;
                        }
                    }
                }

#ifdef KEEP_BATCH_MAXIMUM
                /* Keep track of maximum for this batch */
                if (mpz_cmp(big_max_value, batch_max_value) > 0) {
                    mpz_set(batch_max_value, big_max_value);
                    batch_max_value_n = start;
                }
#endif
            }

            /* Display stuff when needed */
            if (show_it || show_anyway || list_all) {
                if (step_new_record) {
                    printf("N=%lu,\t*STEPS=%u,\t", start, steps);
                } else {
                    printf("N=%lu,\t STEPS=%u,\t", start, steps);
                }

                if (use_gcc_loop) {
                    if (max_new_record) {
                        counter = 0;
                        n_temp  = max_value;
                        while (n_temp > 0UL) {
                            n_temp = n_temp / 10UL;
                            counter++;
                        }
                        printf("*MAX=%lu\t(%lu)\n", max_value, counter);
                    } else {
                        printf(" MAX=%lu\n", max_value);
                    }
                } else { /* using GMP */
                    if (max_new_record) {
                        printf("*MAX=");
                        counter = 0;
                        mpz_set(big_n_temp, big_max_value);
                        while (mpz_sgn(big_n_temp) > 0) { /* Return +1 if op > 0, 0 if op = 0, and -1 if op < 0 */
                            mpz_div_ui(big_n_temp, big_n_temp, 10UL);
                            counter++;
                        }

                        mpz_out_str(stdout, 10, big_max_value);
                        if (show_gmp_flag) {
                            printf("\t(%lu)\t%s\n", counter, gmptxt);
                        } else {
                            printf("\t(%lu)\n", counter);
                        }
                    } else {
                        printf(" MAX=");
                        mpz_out_str(stdout, 10, big_max_value);
                        if (show_gmp_flag) {
                            printf("\t\t%s\n", gmptxt);
                        } else {
                            printf("\n");
                        }
                    }
                }
                fflush(stdout);
                show_anyway=false;
            }

            /* Show progress timer */
            if (i > limit_i) {

                timer_stop   = currentTimeMillis();
                time_elapsed = timer_stop - timer_start;
                timer_start  = timer_stop;
                printf("Now at N=%lu (%ld milliseconds for %u calculations) gmp used %u times avg %lu steps\n", start, time_elapsed, limit_i, gmp_used, steps_bucket/limit_i);

                fflush(stdout);
                i=1U;
                steps_bucket = 0UL;
                gmp_used = 0;
            }

            if (start >= stop) {
                if (use_gcc_loop) {
                    printf("\ngcc mode / Terminating calculation at\nN=%lu,\t STEPS=%u,\t MAX=%lu\n", start, steps, max_value);
                } else {
                    printf("\ngmp mode / Terminating calculation at\nN=%lu,\t STEPS=%u,\t MAX=", start, steps);
                    mpz_out_str(stdout, 10, big_max_value);
                    printf("\t%s\n", gmptxt);
                }

                if (mpz_cmp_ui(big_record_top, record_top) < 0) {
                    printf("\nRecords are:\nSTEPS=%lu\tMAX=%lu\n\n", record_steps, record_top);
                } else {
                    printf("\nRecords are:\nSTEPS=%lu\tMAX=", record_steps);
                    mpz_out_str(stdout, 10, big_record_top);
                    printf("\n\n");
                }

#ifdef KEEP_BATCH_MAXIMUM
                /* Show highest maximum reached in this batch
                   Show nothing if GMP was never used */
                if (mpz_cmp_ui(batch_max_value, 1UL) > 0) {
                    counter = 0;
                    mpz_set(big_n_temp, batch_max_value);
                    while (mpz_sgn(big_n_temp) > 0) {
                        mpz_div_ui(big_n_temp, big_n_temp, 10UL);
                        counter++;
                    }
                    printf("Max value reached in this batch at N=%lu\tMAX=", batch_max_value_n);
                    mpz_out_str(stdout, 10, batch_max_value);
                    printf("\t(%lu)\n\n", counter);
                }
#endif
                run=false;
                fflush(stdout);
            }

            if (start < stop) {
                start += inc;
            }
            i++;
            step_new_record = false;
            max_new_record  = false;
            /* Go back to gcc math */
            use_gcc_loop    = true;
        } /* continue at next n */
    } /* End of !use_gmp */


    /* ************************************* */
    /* Do full computation using gmp library */
    /* ************************************* */

    if (use_gmp) {

        /* Adjust <stop> to be hit by the increment */
        mpz_sub(big_difference, big_stop, big_start);
        remainder = mpz_fdiv_r_ui(big_remainder, big_difference, inc);
        if (remainder > 0UL) {
            mpz_add_ui(big_stop, big_stop, (inc - remainder));
        }

        while (mpz_cmp(big_stop, big_start) >= 0) {
            mpz_set(big_n, big_start);
            mpz_set(big_max_value, big_one);

            steps = 0U;
            show_it = false;

            if (show_steps) {
                printf("    Steps for calculation of n=");
                mpz_out_str(stdout, 10, big_n);
                printf("\n");
            }

            while (mpz_cmp(big_n, big_one) > 0) {
                /* if (mpz_divisible_ui_p(big_n, 2UL)) {*/ /* n is even */
                if (mpz_divisible_2exp_p(big_n, 1)) { /* Is big_n divisible by 2^1 ? */
                    /* Division by 2 */
                    /* mpz_divexact_ui(big_n, big_n, 2UL); */

                    /* Division by 2 doing 1 bitshift right; is a bit faster */
                    mpz_tdiv_q_2exp(big_n, big_n, 1);
                } else { /* n is odd */
                    mpz_mul(big_n, big_n, big_three);
                    mpz_add(big_n, big_n, big_one);
                }

                steps++;

                if (show_steps) {
                    printf("    Step %u:\tn=", steps);
                    mpz_out_str(stdout, 10, big_n);

                    /* Count the number of digits */
                    counter = 0;
                    mpz_set(big_n_temp, big_n);
                    /* Declaration is: int mpz_sgn (const mpz_t op)		*/
                    /* Return +1 if op > 0, 0 if op = 0, and -1 if op < 0.	*/
                    while (mpz_sgn(big_n_temp) > 0) {
                        mpz_div_ui(big_n_temp, big_n_temp, 10UL);
                        counter++;
                    }
                    printf("\t(%lu)\n", counter);
                }

                /* Remember the record for n */
                if (mpz_cmp(big_n, big_max_value) > 0) {
                    mpz_set(big_max_value, big_n);
                }
            }
                
            steps_bucket += steps;
            if (steps > record_steps) {
                show_it         = true;
                record_steps    = steps;
                step_new_record = true;
            }

            if (mpz_cmp(big_max_value, big_record_top) > 0) {
                show_it    = true;
                mpz_set(big_record_top, big_max_value);
                max_new_record = true;
            }

#ifdef KEEP_BATCH_MAXIMUM
            /* Keep track of maximum for this batch */
            if (mpz_cmp(big_max_value, batch_max_value) > 0) {
                mpz_set(batch_max_value, big_max_value);
                mpz_set(big_batch_max_value_n, big_start);
            }
#endif

            if (show_it | list_all) {
                if (step_new_record) {
                    printf("N=");
                    mpz_out_str(stdout, 10, big_start);
                    printf(",\t*STEPS=%u,\t", steps);
                } else {
                    printf("N=");
                    mpz_out_str(stdout, 10, big_start);
                    printf(",\t STEPS=%u,\t", steps);
                }

                if (max_new_record) {
                    printf("*MAX=");
                    mpz_out_str(stdout, 10, big_max_value);

                    /* Count the number of digits */
                    counter = 0;
                    mpz_set(big_n_temp, big_max_value);
                    /* Declaration is: int mpz_sgn (const mpz_t op)		*/
                    /* Return +1 if op > 0, 0 if op = 0, and -1 if op < 0.	*/
                    while (mpz_sgn(big_n_temp) > 0) {
                        mpz_div_ui(big_n_temp, big_n_temp, 10UL);
                        counter++;
                    }
                    printf("\t(%lu)\n", counter);
                } else {
                    printf(" MAX=");
                    mpz_out_str(stdout, 10, big_max_value);
                    printf("\n");
                }
                fflush(stdout);
            }

            /* Show progress timer */
            if (i > limit_i) {
                timer_stop   = currentTimeMillis();
                time_elapsed = timer_stop - timer_start;
                timer_start  = timer_stop;
                printf("Now at N=");
                mpz_out_str(stdout, 10, big_start);
                printf(" (%ld milliseconds for %u calculations) avg %lu steps\n", time_elapsed, limit_i, steps_bucket/limit_i);

                i=1U;
                steps_bucket = 0UL;
                fflush(stdout);
            }

            if (mpz_cmp(big_start, big_stop) == 0) {
                printf("\nGNU MP mode / Terminating calculation at\nN=");
                mpz_out_str(stdout, 10, big_start);
                printf(",\t STEPS=%u,\t MAX=", steps);
                mpz_out_str(stdout, 10, big_max_value);
                printf("\n");
                printf("\nRecords are:\nSTEPS=%lu\tMAX=", record_steps);
                mpz_out_str(stdout, 10, big_record_top);
                printf("\n\n");

#ifdef KEEP_BATCH_MAXIMUM
                /* Show highest maximum reached in this batch
                   Show nothing if GMP was never used */
                if (mpz_cmp_ui(batch_max_value, 1UL) > 0) {
                    counter = 0;
                    mpz_set(big_n_temp, batch_max_value);
                    while (mpz_sgn(big_n_temp) > 0) {
                        mpz_div_ui(big_n_temp, big_n_temp, 10UL);
                        counter++;
                    }
                    printf("Max value reached in this batch at N=");
                    mpz_out_str(stdout, 10, big_batch_max_value_n);
                    printf("\tMAX=");
                    mpz_out_str(stdout, 10, batch_max_value);
                    printf("\t(%lu)\n\n", counter);
                }
#endif
                fflush(stdout);
            }

            mpz_add_ui(big_start, big_start, inc);
            i++;
            step_new_record = false;
            max_new_record  = false;
        } /* continue at next n */
    } /* End of use gmp */

    /* Cleaning up */
    mpz_clear(big_start);
    mpz_clear(big_stop);
    mpz_clear(big_n);
    mpz_clear(big_n_temp);
    mpz_clear(big_max_value);
#ifdef KEEP_BATCH_MAXIMUM
    mpz_clear(batch_max_value);
    mpz_clear(big_batch_max_value_n);
#endif
    mpz_clear(big_record_top);
    mpz_clear(big_difference);
    mpz_clear(big_remainder);
    mpz_clear(big_one);
    mpz_clear(big_two);
    mpz_clear(big_three);
    fflush(stdout);
    exit (EXIT_SUCCESS);
}
